local RemoveAmount = 5
local MaxDistance = 5.0

local GraffitiPlacedTable = 'electus_graffiti_placed'
local GangsTable = 'electus_gangs'
local GangsMembersTable = 'electus_gangs_members'

GraffitiCaptureCache = GraffitiCaptureCache or {}

function ClampInt(value, min, max)
    value = tonumber(value) or 0
    value = math.floor(value)
    if value < min then return min end
    if value > max then return max end
    return value
end

function DecodeCoords(coordsValue)
    if not coordsValue then return nil end
    if type(coordsValue) == 'table' then
        if coordsValue.x and coordsValue.y and coordsValue.z then
            return { x = tonumber(coordsValue.x), y = tonumber(coordsValue.y), z = tonumber(coordsValue.z) }
        end
        return nil
    end

    if type(coordsValue) == 'string' then
        local decoded = json.decode(coordsValue)
        if type(decoded) == 'table' and decoded.x and decoded.y and decoded.z then
            return { x = tonumber(decoded.x), y = tonumber(decoded.y), z = tonumber(decoded.z) }
        end
    end

    return nil
end

function GetGangIdByIdentifier(identifier)
    if not identifier or identifier == '' then return nil end

    local ownerRow = MySQL.single.await(('SELECT gang_id FROM %s WHERE owner = ? LIMIT 1'):format(GangsTable), { identifier })
    if ownerRow and ownerRow.gang_id then
        return tonumber(ownerRow.gang_id)
    end

    local memberRow = MySQL.single.await(('SELECT gang_id FROM %s WHERE identifier = ? LIMIT 1'):format(GangsMembersTable), { identifier })
    if memberRow and memberRow.gang_id then
        return tonumber(memberRow.gang_id)
    end

    return nil
end

function CacheGraffitiId(graffitiId)
    local id = tonumber(graffitiId)
    if not id then return false end
    if GraffitiCaptureCache[id] then return true end

    local row = MySQL.single.await(
        ('SELECT id, identifier, coords FROM %s WHERE id = ? LIMIT 1'):format(GraffitiPlacedTable),
        { id }
    )

    if not row or not row.identifier then return false end

    local coords = DecodeCoords(row.coords)

    GraffitiCaptureCache[id] = {
        identifier = row.identifier,
        coords = coords
    }

    return true
end

function IsPlayerNearCoords(src, coords, maxDistance)
    if not coords then return true end

    local ped = GetPlayerPed(src)
    if not ped or ped == 0 then return false end

    local p = GetEntityCoords(ped)
    local g = vector3(coords.x or 0.0, coords.y or 0.0, coords.z or 0.0)

    return #(p - g) <= (tonumber(maxDistance) or 5.0)
end

RegisterNetEvent('ws_graffiti_capturehook:server:cacheGraffitiId', function(graffitiId)
    CacheGraffitiId(graffitiId)
end)

RegisterNetEvent('ws_graffiti_capturehook:server:onGraffitiRemoved', function(graffitiId)
    local src = source
    local id = tonumber(graffitiId)
    if not id then return end

    local cached = GraffitiCaptureCache[id]
    if not cached then
        return
    end

    if not IsPlayerNearCoords(src, cached.coords, MaxDistance) then
        return
    end

    local zoneId = exports['electus_gangs']:GetSourceZoneId(src)
    if not zoneId or zoneId <= 0 then return end

    local gangId = GetGangIdByIdentifier(cached.identifier)
    if not gangId or gangId <= 0 then return end

    local amount = ClampInt(RemoveAmount, 1, 100000)

    exports['electus_gangs']:RemoveProgressFromCapture(zoneId, gangId, amount)

    GraffitiCaptureCache[id] = nil
end)
